
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

#include <GL/glut.h>

#include "utility.h"
#include "glUtils.h"

#include "map.h"
#include "globals.h"
#include "oglRender.h"

#include "shader.h"

#define DrawAsPoints	0


char * MapShader =
/* Heightfield shader that handles scaling, texture coord
   calculation from raw height data. */
"!!ARBvp1.0					\n\
# Setup						\n\
OUTPUT	oPos	= result.position;		\n\
OUTPUT  oCol	= result.color;			\n\
OUTPUT	oTex	= result.texcoord;		\n\
PARAM	CTM[4]	= { state.matrix.mvp };		\n\
PARAM	IMV[4]	= { state.matrix.modelview.invtrans };	\n\
PARAM   gScale  = program.env[1];		\n\
PARAM	hRange	= program.env[2];		\n\
PARAM	lVec	= program.env[3];		\n\
TEMP    vPos, vTex;				\n\
TEMP    vec, norm, eyeNorm, coeff, shade;	\n\
PARAM	ZERO = { 0, 0, 0, 0 };			\n\
PARAM	ONE  = { 1, 1, 1, 1 };			\n\
PARAM   NEG  = { -1, -1, -1, -1 };		\n\
## Texture coord from height			\n\
# Outgoing Q must be 1, others don't care	\n\
MOV  vTex, ONE;					\n\
MOV  vTex.x, vertex.position.y;			\n\
MUL  vTex, vTex, hRange;			\n\
MOV  oTex, vTex;				\n\
## Scale heightfield data			\n\
MOV  vPos, vertex.position;			\n\
MUL  vPos, vPos, gScale;			\n\
## Calc surface normal by taking vectors to each\n\
## neighbouring height (passed in as tex coords)\n\
## and averaging				\n\
MOV  norm, ZERO;				\n\
# Height at point -1, 0				\n\
MOV  vec, vertex.position;			\n\
SUB  vec.x, vec.x, ONE.x;			\n\
MOV  vec.y, vertex.texcoord.x;			\n\
SUB  vec, vec, vertex.position;			\n\
# Scale to current map and normalise		\n\
MUL  vec, vec, gScale;				\n\
DP3  vec.w, vec, vec;				\n\
RSQ  vec.w, vec.w;				\n\
MUL  vec, vec, vec.w;				\n\
# Accumulate					\n\
ADD  norm, norm, vec;				\n\
# Height at point +1, 0				\n\
MOV  vec, vertex.position;			\n\
ADD  vec.x, vec.x, ONE.x;			\n\
MOV  vec.y, vertex.texcoord.y;			\n\
SUB  vec, vec, vertex.position;			\n\
MUL  vec, vec, gScale;				\n\
DP3  vec.w, vec, vec;				\n\
RSQ  vec.w, vec.w;				\n\
MUL  vec, vec, vec.w;				\n\
ADD  norm, norm, vec;				\n\
# Height at point 0, -1				\n\
MOV  vec, vertex.position;			\n\
SUB  vec.z, vec.z, ONE.z;			\n\
MOV  vec.y, vertex.texcoord.z;			\n\
SUB  vec, vec, vertex.position;			\n\
MUL  vec, vec, gScale;				\n\
DP3  vec.w, vec, vec;				\n\
RSQ  vec.w, vec.w;				\n\
MUL  vec, vec, vec.w;				\n\
ADD  norm, norm, vec;				\n\
# Height at point 0, +1				\n\
MOV  vec, vertex.position;			\n\
ADD  vec.z, vec.z, ONE.z;			\n\
MOV  vec.y, vertex.texcoord.w;			\n\
SUB  vec, vec, vertex.position;			\n\
MUL  vec, vec, gScale;				\n\
DP3  vec.w, vec, vec;				\n\
RSQ  vec.w, vec.w;				\n\
MUL  vec, vec, vec.w;				\n\
ADD  norm, norm, vec;				\n\
# Final eye space surface normal		\n\
DP3  eyeNorm.x, IMV[0], norm;			\n\
DP3  eyeNorm.y, IMV[1], norm; 			\n\
DP3  eyeNorm.z, IMV[2], norm;			\n\
DP3  eyeNorm.w, eyeNorm, eyeNorm;		\n\
RSQ  eyeNorm.w, eyeNorm.w;			\n\
MUL  eyeNorm, eyeNorm, eyeNorm.w;		\n\
## Standard transform				\n\
DP4  oPos.x, CTM[0], vPos;			\n\
DP4  oPos.y, CTM[1], vPos;			\n\
DP4  oPos.z, CTM[2], vPos;			\n\
DP4  oPos.w, CTM[3], vPos;			\n\
## Lighting					\n\
## Ambient: no setup required			\n\
MUL  shade, state.lightmodel.ambient, ONE;	\n\
## Diffuse: light dot normal			\n\
DP3  coeff.x, lVec, eyeNorm;			\n\
## No specular					\n\
MOV  coeff.y, ZERO.y;				\n\
MOV  coeff.w, ZERO.w;				\n\
# Eval coefficients and sum			\n\
LIT  coeff, coeff;				\n\
MAD  shade, coeff.y, ONE, shade;		\n\
MAD  shade, coeff.z, ONE, shade;		\n\
MOV  oCol, shade;				\n\
END\n\
";

static GLuint Shader;

#define MaxHeight	255.0

static GLfloat * data = NULL;
static int	 nTris;

static void genHeight (GLfloat * data, MapRec * map, int x, int z)
{
  data[0] = GetMap(map, x - 1, z);
  data[1] = GetMap(map, x + 1, z);
  data[2] = GetMap(map, x, z - 1);
  data[3] = GetMap(map, x, z + 1);
  
  data[4] = x;
  data[5] = GetMap(map, x, z);
  data[6] = z;
  data[7] = 1; // w
}

static void genTriStrip (GLfloat buf[], MapRec * map)
{
  int idx, x, z;
  
  idx = 0;
  for (z = 1; z < map->depth - 2; z++) {
    for (x = 1; x < map->width - 1; x++) {
      genHeight(&buf[idx], map, x, z); idx += 8;
      genHeight(&buf[idx], map, x, z + 1); idx += 8;
    }
  }
  nTris = (map->width - 2) * (map->depth - 3) * 2;
}

void RenderShader (MapRec * map)
{
  GLfloat   scale[4];
  GLfloat   range[4];
  GLfloat   light[4];
  int	    z, w, h;
  
  glEnable(GL_VERTEX_PROGRAM_ARB);
  glEnable(GL_LIGHTING);
  glEnable(GL_TEXTURE_1D);
  
  glDisable(GL_LIGHTING);

  if (data == NULL) {
    /* GL_T4F_V4F format array */
    data = New(sizeof(GLfloat) * map->width * map->depth * 2 * 8);
    App->recalc = TRUE;
  }
  
  if (App->recalc) {
    genTriStrip(data, map);
    App->recalc = FALSE;
  }
  
  scale[0] = map->groundScale;
  scale[1] = map->heightScale;
  scale[2] = map->groundScale;
  scale[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 1, scale);
  
  range[0] = 1.0 / MaxHeight;
  range[1] = 0;
  range[2] = 0;
  range[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 2, range);
  
  glEnable(GL_LIGHTING);
  SetLightPosition();
  /* Pass light vector to vertex shader. Not strictly necessary
     as the position is part of the program state, but this way
     the vertex shader doesn't have to normalize it each time. */
  glGetLightfv(GL_LIGHT0, GL_POSITION, light);
  NormVec(light);
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 3, light);
  
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);
  glInterleavedArrays(GL_T4F_V4F, 0, data);
  if (DrawAsPoints) {
    glDisable(GL_LIGHTING);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glDrawArrays(GL_POINTS, 0, nTris);
  } else {
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    w = map->width - 2;
    h = map->depth - 3;
    for (z = 0; z < h; z++)
      glDrawArrays(GL_TRIANGLE_STRIP,
      		z * w * 2,
      		w * 2);
  }
  
  glPopMatrix();
  glDisable(GL_VERTEX_PROGRAM_ARB);
  glDisable(GL_TEXTURE_1D);
  glDisable(GL_LIGHTING);
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

void InitShader ()
{
  int  err, i;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
    
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  glGenProgramsARB(1, &Shader);
  
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader);
  glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			strlen(MapShader), MapShader);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0) {
    getErrorLine(MapShader, err, msg, sizeof(msg));
    printf("Error in shader #%d: %s\n", i,
      	      glGetString(GL_PROGRAM_ERROR_STRING_ARB));
    printf("Source line:    %s\n", msg);
  }
  
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader);
  				
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

