
#include <stdio.h>
#include <math.h>

#include <GL/glut.h>

#include "utility.h"
#include "glUtils.h"

#include "map.h"
#include "globals.h"

#include "render.h"

char * MapShader =
/* Heightfield shader that handles scaling, texture coord
   calculation from raw height data. */
"!!ARBvp1.0					\n\
# Setup						\n\
OUTPUT	oPos	= result.position;		\n\
OUTPUT  oCol	= result.color;			\n\
OUTPUT	oTex	= result.texcoord;		\n\
OUTPUT  oFog	= result.fogcoord;		\n\
PARAM	CTM[4]	= { state.matrix.mvp };		\n\
PARAM   gScale  = program.env[1];		\n\
PARAM	hRange	= program.env[2];		\n\
PARAM	fogAdj	= program.env[3];		\n\
TEMP    vPos, vTex, vFog;			\n\
PARAM	ZERO = { 0, 0, 0, 0 };			\n\
PARAM	ONE  = { 1, 1, 1, 1 };			\n\
PARAM   NEG  = { -1, -1, -1, -1 };		\n\
## Texture coord from height			\n\
# Outgoing Q must be 1, others don't care	\n\
MOV  vTex, ONE;					\n\
MOV  vTex.x, vertex.position.y;			\n\
MUL  vTex, vTex, hRange;			\n\
MOV  oTex, vTex;				\n\
## Scale heightfield data			\n\
MOV  vPos, vertex.position;			\n\
MUL  vPos, vPos, gScale;			\n\
## Fog value based on Y				\n\
MOV  vFog, ONE;					\n\
SUB  vFog.x, vFog.x, vTex.x;			\n\
ADD  vFog, vFog, fogAdj;			\n\
MOV  oFog, vFog;				\n\
## Standard transform				\n\
DP4  oPos.x, CTM[0], vPos;			\n\
DP4  oPos.y, CTM[1], vPos;			\n\
DP4  oPos.z, CTM[2], vPos;			\n\
DP4  oPos.w, CTM[3], vPos;			\n\
## In case we forgot to change texture mode	\n\
MOV  oCol, ONE;					\n\
END\n\
";


#define MaxHeight	255.0

static GLfloat * verts = NULL;
static GLfloat * tex   = NULL;
static GLfloat * norms = NULL;
static int	 nVerts= 0;

static GLuint	 buf = 0;
static GLuint	 texMap = 0;


/****		Texture map used for height color scale	****/

static GLubyte Rainbow[8][3] = {
 { 0x3F, 0x00, 0x3F },
 { 0x7F, 0x00, 0x7F },
 { 0xBF, 0x00, 0xBF },
 { 0x00, 0x00, 0xFF },
 { 0x00, 0xFF, 0x00 },
 { 0xFF, 0xFF, 0x00 },
 { 0xFF, 0x7F, 0x00 },
 { 0xFF, 0x00, 0x00 }
};

static void createTextureTable ()
{
  glGenTextures(1, &texMap);
  glBindTexture(GL_TEXTURE_1D, texMap);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexImage1D(GL_TEXTURE_1D, 0, GL_RGB, 8, 0, GL_RGB, GL_UNSIGNED_BYTE, Rainbow);
  CheckGL();
}


/****		Generate coords for triangle strip	****/

static void setHeight (GLfloat * xyz, MapRec * map, int x, int z)
{
  xyz[0] = (GLfloat)x * map->groundScale;
  xyz[1] = GetMap(map, x, z) * map->heightScale;
  xyz[2] = (GLfloat)z * map->groundScale;
}

static void createTriStrip (GLfloat verts[], MapRec * map)
{
  int idx, x, z;
  
  idx = 0;
  for (x = 0; x < map->width; x++) {
    for (z = 0; z < map->depth - 1; z++) {
      setHeight(&verts[idx], map, x, z); idx ++;
      setHeight(&verts[idx], map, x, z + 1); idx ++;
    }
  }
  nVerts = idx;
}


/****		Debugging: render as points 	****/


static void oglRenderPoints (MapRec * map)
{
  if (verts == NULL)
    verts = New(sizeof(GLfloat) * map->width * map->depth * 6);
  if (tex == NULL) {
    tex = New(sizeof(GLfloat) * map->width * map->depth);
    createTextureTable();
    //calcTexCoords(tex, map);
  }
    
  glEnableClientState(GL_VERTEX_ARRAY);
  //glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glDisable(GL_LIGHTING);
  
  if (App->rescale) {
    /* calcMap(verts, map, map->groundScale, map->heightScale); */
    createTriStrip(verts, map);
    App->rescale = FALSE;
  }
  
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  //glEnable(GL_TEXTURE_1D);
  //glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

  glVertexPointer(3, GL_FLOAT, 0, verts);
  //glTexCoordPointer(1, GL_FLOAT, 0, tex);
  glDrawArrays(GL_POINTS, 0, nVerts);
  
  //glDisable(GL_TEXTURE_1D);
  glPopMatrix();
  glDisableClientState(GL_VERTEX_ARRAY);
  //glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}


static int clamp (int idx, int max)
{
  if (idx < 0)
    return 0;
  else if (idx > max)
    return max;
  else
    return idx;
}

static float getMap (MapRec * map, int i, int j)
{
  i = Clamp(i, 0, map->depth - 1);
  j = Clamp(j, 0, map->width - 1);
  return map->data[i * map->width + j];
}

static void calcMap (GLfloat verts[], MapRec * map, float hScale, float vScale)
{
  int idx, i, j;
  
  idx = 0;
  for (i = 0; i < map->width; i++) {
    for (j = 0; j < map->depth; j++) {
      verts[idx] = (GLfloat)i * hScale; idx ++;
      verts[idx] = getMap(map, i, j) * vScale; idx ++;
      verts[idx] = (GLfloat)j * hScale; idx ++;
    }
  }
}

static void calcTexCoords (GLfloat tex[], MapRec * map)
{
  int   idx, i, j;
  float height;
  
  idx = 0;
  for (i = 0; i < map->width; i++) {
    for (j = 0; j < map->depth; j++) {
      height = getMap(map, i, j);
      tex[idx] = Clamp(height / MaxHeight, 0, 1);
      idx ++;
    }
  }
}

static void getVec (MapRec * map, int i, int j, int dx, int dz, Vec3f v)
{
  float h1, h2;
  
  h1 = getMap(map, i, j);
  h2 = getMap(map, i + dx, j + dz);
  v[0] = dx * map->groundScale;
  v[1] = h2 - h1;
  v[2] = dz * map->groundScale;
}

static void vecSet (Vec3f dest, GLfloat x, GLfloat y, GLfloat z)
{
  dest[0] = x;
  dest[1] = y;
  dest[2] = z;
}

static void vecAdd (Vec3f dest, Vec3f v1, Vec3f v2)
{
  int i;
  
  for (i = 0; i < 3; i++)
    dest[i] = v1[i] + v2[i];
}

static void vecNorm (Vec3f v)
{
  float len;
  int   i;
  
  len = sqrtf(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  if (len > 0)
    for (i = 0; i < 3; i++)
      v[i] = v[i] / len;
}

static void calcNormals (GLfloat norms[], MapRec * map)
{
  int   idx, i, j, k;
  Vec3f v[4];
  Vec3f sum;
  
  idx = 0;
  for (i = 0; i < map->width; i++) {
    for (j = 0; j < map->depth; j++) {
      /* Difference from four neighbouring heights */
      getVec(map, i, j, -1, 0, v[0]);
      getVec(map, i, j, +1, 0, v[1]);
      getVec(map, i, j, 0, -1, v[2]);
      getVec(map, i, j, 0, +1, v[3]);
      /* Average is normal */
      vecSet(sum, 0, 0, 0);
      for (k = 0; k < 4; k++) {
        vecNorm(v[k]);
        vecAdd(sum, sum, v[k]);
      }
      vecNorm(sum);
      /* And store */
      norms[idx] = sum[0]; idx ++;
      norms[idx] = sum[1]; idx ++;
      norms[idx] = sum[2]; idx ++;
    }
  }
}

/****		Standard OpenGL vertex array		****/


static void oglRender (MapRec * map)
{
  int j;
  
  if (verts == NULL)
    verts = New(sizeof(GLfloat) * map->width * map->depth * 6);
  if (tex == NULL) {
    tex = New(sizeof(GLfloat) * map->width * map->depth * 2);
    //createTextureTable();
    //calcTexCoords(tex, map);
  }
  if (norms == NULL)
    norms = New(sizeof(GLfloat) * map->width * map->depth * 6);
    
  if (App->rescale) {
    createTriStrip(verts, map);
    //calcNormals(norms, map);
    App->rescale = FALSE;
  }
  
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  //glEnable(GL_TEXTURE_1D);
  //glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  for (j = 0; j < map->depth - 1; j++) {
    printf("Strip #%d\n", j);
    glVertexPointer(3, GL_FLOAT, 0, &(verts[j * map->width * 6]));
    glDrawArrays(GL_TRIANGLE_STRIP, 0, map->width * 2);
  }
  
  //glVertexPointer(3, GL_FLOAT, 0, verts);
  //glTexCoordPointer(1, GL_FLOAT, 0, tex);
  //glNormalPointer(GL_FLOAT, 0, norms);
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  //glDisable(GL_TEXTURE_1D);
  glPopMatrix();
}


/****		Render with buffer object		****/


static void initBuffer (MapRec * map)
{
  glGenBuffersARB(1, &buf);
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  glBufferDataARB(GL_ARRAY_BUFFER_ARB,
  		sizeof(GLfloat) * map->width * map->depth * 4, // xyz + s
		NULL, GL_DYNAMIC_DRAW_ARB);
}

static void vboRender (MapRec * map)
{
  GLfloat * verts;
  GLfloat * tex;
  
  if (buf == 0)
    initBuffer(map);
  
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  if (App->rescale) {
    verts = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY);
    calcMap(verts, map, map->groundScale, map->heightScale);
    // Strictly speaking we only need to do this once, but the logic
    // is messy enough already that I don't want to bother.
    tex = verts + (map->width * map->depth * 3);
    calcTexCoords(tex, map);
    if (! glUnmapBufferARB(GL_ARRAY_BUFFER_ARB))
      Fail("glUnmapBufferARB");
    App->rescale = FALSE;
  }
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  glVertexPointer(3, GL_FLOAT, 0, 0);
  glTexCoordPointer(1, GL_FLOAT, 0, (void *)(sizeof(GLfloat) * map->width * map->depth * 3));
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  glPopMatrix();
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
}


/****		Render with shader		****/

static void shaderRender (MapRec * map)
{
  GLfloat * verts;
  GLfloat   scale[4];
  GLfloat   range[4];
  GLfloat   fogAdj[4];
  
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  if (buf == 0) {
    initBuffer(map);
    verts = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY);
    calcMap(verts, map, 1, 1);
    if (! glUnmapBufferARB(GL_ARRAY_BUFFER_ARB))
      Fail("glUnmapBufferARB");
    App->rescale = TRUE;
  }
  
  scale[0] = map->groundScale;
  scale[1] = map->heightScale;
  scale[2] = map->groundScale;
  scale[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 1, scale);
  
  range[0] = 1.0 / MaxHeight;
  range[1] = 0;
  range[2] = 0;
  range[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 2, range);
  
  fogAdj[0] = (App->viewHeight - MaxHeight) / MaxHeight * 0.5;
  fogAdj[1] = 0;
  fogAdj[2] = 0;
  fogAdj[3] = 1;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 3, fogAdj);
  
  App->rescale = FALSE;

  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);
  glVertexPointer(3, GL_FLOAT, 0, 0);
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  glPopMatrix();
  glDisable(GL_VERTEX_PROGRAM_ARB);
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
}

void RenderMap (MapRec * map)
{
  if (App->useShader)
    shaderRender(map);
  else
    oglRenderPoints(map);
}

static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };
static GLfloat LightPos[4] = { 0, 1, 4, 0 };

void InitRender ()
{
  glShadeModel(GL_SMOOTH);
  glEnable(GL_RESCALE_NORMAL);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, Black);
  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
}
  if (z < 0)
    z = 0;
  else if (z >= map->depth - 1;
    z = map->depth - 1;
