
#include <stdio.h>
#include <math.h>

#include <GL/glut.h>

#include "utility.h"
#include "glUtils.h"

#include "map.h"
#include "globals.h"

#include "oglRender.h"

#define MaxHeight	255.0

#define DrawAsPoints	0

static GLfloat * verts	= NULL;
static GLfloat * tex	= NULL;
static GLfloat * norms	= NULL;
static int	 nTris	= 0;
static GLuint	 texMap	= 0;

static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };
static GLfloat LightPos[4] = { 0, 1, 1, 0 };


/****		Texture map used for height color scale	****/

static GLubyte Rainbow[8][3] = {
 { 0x3F, 0x00, 0x3F },
 { 0x7F, 0x00, 0x7F },
 { 0xBF, 0x00, 0xBF },
 { 0x00, 0x00, 0xFF },
 { 0x00, 0xFF, 0x00 },
 { 0xFF, 0xFF, 0x00 },
 { 0xFF, 0x7F, 0x00 },
 { 0xFF, 0x00, 0x00 }
};

static void createTextureTable ()
{
  glGenTextures(1, &texMap);
  glBindTexture(GL_TEXTURE_1D, texMap);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexImage1D(GL_TEXTURE_1D, 0, GL_RGB, 8, 0, GL_RGB, GL_UNSIGNED_BYTE, Rainbow);
  CheckGL();
}


/****		Geometry for triangle strips	****/


static void genHeight (GLfloat * xyz, MapRec * map, int x, int z)
{
  xyz[0] = (GLfloat)x * map->groundScale;
  xyz[1] = GetMap(map, x, z) * map->heightScale;
  xyz[2] = (GLfloat)z * map->groundScale;
}

static void genTriVerts (GLfloat verts[], MapRec * map)
{
  int idx, x, z;
  
  idx = 0;
  for (z = 1; z < map->depth - 2; z++) {
    for (x = 1; x < map->width - 1; x++) {
      genHeight(&verts[idx], map, x, z); idx += 3;
      genHeight(&verts[idx], map, x, z + 1); idx += 3;
    }
  }
  nTris = (map->width - 2) * (map->depth - 3) * 2;
}

static void genTexCoords (GLfloat tex[], MapRec * map)
{
  int idx, x, z;
  
  idx = 0;
  for (z = 1; z < map->depth - 2; z++) {
    for (x = 1; x < map->width - 1; x++) {
      tex[idx] = Clamp(GetMap(map, x, z) / MaxHeight, 0, 1); idx ++;
      tex[idx] = Clamp(GetMap(map, x, z + 1) / MaxHeight, 0, 1); idx ++;
    }
  }
}

/* Surface normal calculation: much more complex :-( */

static void calcDiffVec (MapRec * map, int x, int z, int dx, int dz, Vec3f diff)
{
  float h1, h2;
  
  h1 = GetMap(map, x, z);
  h2 = GetMap(map, x + dx, z + dz);
  diff[0] = dx * map->groundScale;
  diff[1] = (h2 - h1) * map->heightScale;
  diff[2] = dz * map->groundScale;
}

static void genAverageNormal (MapRec * map, int x, int z, Vec3f norm)
{
  Vec3f v[4];
  int   i;
  
  /* Difference from four neighbouring heights */
  calcDiffVec(map, x, z, -1, 0, v[0]);
  calcDiffVec(map, x, z, +1, 0, v[1]);
  calcDiffVec(map, x, z, 0, -1, v[2]);
  calcDiffVec(map, x, z, 0, +1, v[3]);
  /* Average is normal */
  SetVec(norm, 0, 0, 0);
  for (i = 0; i < 4; i++) {
    NormVec(v[i]);
    AddVec(norm, norm, v[i]);
  }
  NormVec(norm);
}

static void genNormals (GLfloat norm[], MapRec * map)
{
  int   idx, x, z;
  Vec3f v;
  
  idx = 0;
  for (z = 1; z < map->depth - 2; z++) {
    for (x = 1; x < map->width - 1; x++) {
      genAverageNormal(map, x, z, v);
      norms[idx] = v[0]; idx ++;
      norms[idx] = v[1]; idx ++;
      norms[idx] = v[2]; idx ++;
      genAverageNormal(map, x, z + 1, v);
      norms[idx] = v[0]; idx ++;
      norms[idx] = v[1]; idx ++;
      norms[idx] = v[2]; idx ++;
    }
  } 
}

void SetLightPosition ()
{
  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
}

void RenderGL (MapRec * map)
{
  int z, w, h;
  
  if (verts == NULL)
    verts = New(sizeof(GLfloat) * map->width * map->depth * 6);
  if (tex == NULL) {
    tex = New(sizeof(GLfloat) * map->width * map->depth * 2);
    createTextureTable();
    genTexCoords(tex, map);
  }
  if (norms == NULL)
    norms = New(sizeof(GLfloat) * map->width * map->depth * 6);
    
  if (App->recalc) {
    genTriVerts(verts, map);
    genNormals(norms, map);
    App->recalc = FALSE;
  }
  
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  glEnable(GL_LIGHTING);
  SetLightPosition();
  glEnable(GL_TEXTURE_1D);
  
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  
  glVertexPointer(3, GL_FLOAT, 0, verts);
  glTexCoordPointer(1, GL_FLOAT, 0, tex);
  glNormalPointer(GL_FLOAT, 0, norms);

  if (DrawAsPoints) {
    glDisable(GL_LIGHTING);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glDrawArrays(GL_POINTS, 0, nTris);
  } else {
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    glColor3f(1, 1, 1);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    w = map->width - 2;
    h = map->depth - 3;
    for (z = 0; z < h; z++)
      glDrawArrays(GL_TRIANGLE_STRIP,
      		z * w * 2,
      		w * 2);
  }
  
  glDisableClientState(GL_VERTEX_ARRAY);
  glDisableClientState(GL_TEXTURE_COORD_ARRAY);
  glDisableClientState(GL_NORMAL_ARRAY);
  
  glDisable(GL_TEXTURE_1D);
  glDisable(GL_LIGHTING);
  glPopMatrix();
}


void InitRenderGL ()
{
  glShadeModel(GL_SMOOTH);
  glEnable(GL_RESCALE_NORMAL);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, Black);
}

GLuint GetHeightTexMap ()
{
  return texMap;
}
