
#include <stdio.h>

#include <GL/glut.h>

#include "utility.h"
#include "glUtils.h"

#include "map.h"
#include "globals.h"

#include "render.h"

char * MapShader =
/* Heightfield shader that handles scaling, texture coord
   calculation from raw height data. */
"!!ARBvp1.0					\n\
# Setup						\n\
OUTPUT	oPos	= result.position;		\n\
OUTPUT  oCol	= result.color;			\n\
OUTPUT	oTex	= result.texcoord;		\n\
PARAM	CTM[4]	= { state.matrix.mvp };		\n\
PARAM   gScale  = program.env[1];		\n\
PARAM	hRange	= program.env[2];		\n\
TEMP    vPos, vTex;				\n\
PARAM	ZERO = { 0, 0, 0, 0 };			\n\
PARAM	ONE  = { 1, 1, 1, 1 };			\n\
## Texture coord from height			\n\
# Outgoing Q must be 1, others don't care	\n\
MOV  vTex, ONE;					\n\
MOV  vTex.x, vertex.position.y;			\n\
MUL  vTex, vTex, hRange;			\n\
MOV  oTex, vTex;				\n\
## Scale heightfield data			\n\
MOV  vPos, vertex.position;			\n\
MUL  vPos, vPos, gScale;			\n\
## Standard transform				\n\
DP4  oPos.x, CTM[0], vPos;			\n\
DP4  oPos.y, CTM[1], vPos;			\n\
DP4  oPos.z, CTM[2], vPos;			\n\
DP4  oPos.w, CTM[3], vPos;			\n\
## In case we forgot to change texture mode	\n\
MOV  oCol, ONE;					\n\
END\n\
";


#define MaxHeight	255.0

static GLfloat * verts = NULL;
static GLfloat * tex   = NULL;

static GLuint	 buf = 0;
static GLuint	 texMap = 0;
static int	 lastTech = -1;

static void calcMap (GLfloat verts[], MapRec * map, float hScale, float vScale)
{
  int idx, x, z;
  
  idx = 0;
  for (x = 0; x < map->width; x++) {
    for (z = 0; z < map->depth; z++) {
      verts[idx] = (GLfloat)x * hScale; idx ++;
      verts[idx] = GetMap(map, x, z) * vScale; idx ++;
      verts[idx] = (GLfloat)z * hScale; idx ++;
    }
  }
}

static void calcTexCoords (GLfloat tex[], MapRec * map)
{
  int idx, x, z;
  float height;
  
  idx = 0;
  for (x = 0; x < map->width; x++) {
    for (z = 0; z < map->depth; z++) {
      height = GetMap(map, x, z);
      tex[idx] = Clamp(height / MaxHeight, 0, 1);
      idx ++;
    }
  }
}

static GLubyte Rainbow[8][3] = {
 { 0x3F, 0x00, 0x3F },
 { 0x7F, 0x00, 0x7F },
 { 0xBF, 0x00, 0xBF },
 { 0x00, 0x00, 0xFF },
 { 0x00, 0xFF, 0x00 },
 { 0xFF, 0xFF, 0x00 },
 { 0xFF, 0x7F, 0x00 },
 { 0xFF, 0x00, 0x00 }
};

static void createTextureTable ()
{
  glGenTextures(1, &texMap);
  glBindTexture(GL_TEXTURE_1D, texMap);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexImage1D(GL_TEXTURE_1D, 0, GL_RGB, 8, 0, GL_RGB, GL_UNSIGNED_BYTE, Rainbow);
  CheckGL();
}

/****		Standard OpenGL vertex array		****/


static void oglRender (MapRec * map)
{
  if (verts == NULL)
    verts = New(sizeof(GLfloat) * map->width * map->depth * 3);
  if (tex == NULL) {
    tex = New(sizeof(GLfloat) * map->width * map->depth);
    createTextureTable();
    calcTexCoords(tex, map);
  }
  
  if (App->rescale) {
    calcMap(verts, map, map->groundScale, map->heightScale);
    App->rescale = FALSE;
  }
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  glVertexPointer(3, GL_FLOAT, 0, verts);
  glTexCoordPointer(1, GL_FLOAT, 0, tex);
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  glPopMatrix();
}


/****		Render with buffer object		****/


static void initBuffer (MapRec * map)
{
  glGenBuffersARB(1, &buf);
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  glBufferDataARB(GL_ARRAY_BUFFER_ARB,
  		sizeof(GLfloat) * map->width * map->depth * 4, // xyz + s
		NULL, GL_DYNAMIC_DRAW_ARB);
}

static void vboRender (MapRec * map)
{
  GLfloat * verts;
  GLfloat * tex;
  
  if (buf == 0)
    initBuffer(map);
  
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  if (App->rescale) {
    verts = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY);
    calcMap(verts, map, map->groundScale, map->heightScale);
    // Strictly speaking we only need to do this once, but the logic
    // is messy enough already that I don't want to bother.
    tex = verts + (map->width * map->depth * 3);
    calcTexCoords(tex, map);
    if (! glUnmapBufferARB(GL_ARRAY_BUFFER_ARB))
      Fail("glUnmapBufferARB");
    App->rescale = FALSE;
  }
  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);

  glVertexPointer(3, GL_FLOAT, 0, 0);
  glTexCoordPointer(1, GL_FLOAT, 0, (void *)(sizeof(GLfloat) * map->width * map->depth * 3));
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  glPopMatrix();
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
}


/****		Render with shader		****/

static void shaderRender (MapRec * map)
{
  GLfloat * verts;
  GLfloat   scale[4];
  GLfloat   range[4];
  
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, buf);
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  if (buf == 0) {
    initBuffer(map);
    verts = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY);
    calcMap(verts, map, 1, 1);
    if (! glUnmapBufferARB(GL_ARRAY_BUFFER_ARB))
      Fail("glUnmapBufferARB");
    App->rescale = TRUE;
  }
  
  scale[0] = map->groundScale;
  scale[1] = map->heightScale;
  scale[2] = map->groundScale;
  scale[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 1, scale);
  
  range[0] = 1.0 / MaxHeight;
  range[1] = 0;
  range[2] = 0;
  range[3] = 1.0;
  glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 2, range);
  
  App->rescale = FALSE;

  glPushMatrix();
  glTranslatef(-map->width * map->groundScale * 0.5,
  		0,
  		-map->depth * map->groundScale * 0.5);
  glVertexPointer(3, GL_FLOAT, 0, 0);
  glDrawArrays(GL_POINTS, 0, map->width * map->depth);
  
  glPopMatrix();
  glDisable(GL_VERTEX_PROGRAM_ARB);
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
}

void RenderMap (MapRec * map, int techLevel)
{
  if (techLevel != lastTech) {
    App->rescale = TRUE;
  }
  
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  switch (techLevel) {
    case renderOGL:
      oglRender(map);
      break;
    case renderBufferObj:
      vboRender(map);
      break;
    case renderShader:
      shaderRender(map);
      break;
    default:
      Fail("Unknown tech level code in RenderMap");
  }
  glDisableClientState(GL_VERTEX_ARRAY);
  glDisableClientState(GL_TEXTURE_COORD_ARRAY);
  lastTech = techLevel;
}

void InitRender ()
{
  glEnable(GL_TEXTURE_1D);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
}
