
/*	Hugh Fisher 2004.

	Vertex shader to do heightfield rendering. Demonstrates
	that geometry can be transformed very efficiently on
	the GPU instead of the host CPU.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#include "map.h"
#include "globals.h"
#include "inputs.h"
#include "render.h"

SharedData *	App;

#define cmdStdOpenGL	 1
#define cmdBufferObj	 2
#define cmdVertexShader	 3
#define cmdExit		99

static int	Menu;
static GLuint	Shader;

/****		Drawing the world		****/

static void setViewpoint ()
{
  gluLookAt(0, App->viewHeight, App->viewDist,
  	    0, App->viewHeight - 0.25, App->viewDist - 1,
	    0, 1, 0);
}

static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  setViewpoint();
  
  glPushMatrix();
    glRotatef(App->viewRot, 0, 1, 0);
    RenderMap(&App->map, App->render);
  glPopMatrix();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  App->winWidth  = width;
  App->winHeight = height;
  
  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, App->winNear, App->winFar);
  glMatrixMode(GL_MODELVIEW);
}


static void menuChoice (int item)
{
  switch (item) {
    case cmdStdOpenGL:
      App->render = renderOGL;
      printf("Standard OpenGL\n");
      break;
    case cmdBufferObj:
      App->render = renderBufferObj;
      printf("Vertex buffer object\n");
      break;
    case cmdVertexShader:
      App->render = renderShader;
      printf("Vertex shader\n");
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}


/****		Main control		****/


static void initGraphics ()
{
  if (! glutExtensionSupported("GL_ARB_vertex_buffer_object"))
    Fail("GL_ARB_vertex_buffer_object not available on this machine");
    
  glClearColor(0, 0, 0, 0); 
  glEnable(GL_DEPTH_TEST);
  glShadeModel(GL_SMOOTH);
  glDisable(GL_LIGHTING);
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

static void initShader ()
{
  int  err, i;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
    
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  glGenProgramsARB(1, &Shader);
  
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader);
  glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			strlen(MapShader), MapShader);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0) {
    getErrorLine(MapShader, err, msg, sizeof(msg));
    printf("Error in shader #%d: %s\n", i,
      	      glGetString(GL_PROGRAM_ERROR_STRING_ARB));
    printf("Source line:    %s\n", msg);
  }
  
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader);
  				
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

static void initApp ()
{
  App->winNear	= 1;
  App->winFar	= 5000;
  App->animate  = TRUE;
  App->render   = renderOGL;
  
  initGraphics();
  initShader();
  InitRender();
  /* Make sure we did everything right */
  CheckGL();
  
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("OpenGL render", cmdStdOpenGL);
  glutAddMenuEntry("Buffer object", cmdBufferObj);
  glutAddMenuEntry("Vertex shader", cmdVertexShader);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
  
  ReadDefaultMap(&App->map);
}

static void processArgs (int argc, char * argv[])
{
  int i;
  char * arg;
  
  i = 1;
  while (i < argc) {
    arg = argv[i];
    if (arg[0] != '-')
      App->map.name = strdup(arg);
    else if (strcasecmp(arg, "-w") == 0) {
      i ++;
      App->map.width = atoi(argv[i]);
    } else if (strcasecmp(arg, "-d") == 0) {
      i ++;
      App->map.depth = atoi(argv[i]);
    } else
      fprintf(stderr, "Unrecognised option: %s\n", arg);
    i ++;
  }
}

static void idle ()
{
  if (App->animate) {
    App->viewRot += 0.1;
    if (App->viewRot >= 360)
      App->viewRot = 0;
    glutPostRedisplay();
  }
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  App = New(sizeof(SharedData));
   
  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("Heightfield vertex program");

  processArgs(argc, argv);
  initApp();
  PrintHelp();
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(idle);
  glutKeyboardFunc(AsciiKey);
  glutSpecialFunc(SpecialKey);
  glutMouseFunc(MouseButton);
  glutMotionFunc(MouseMotion);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
