
/*	Hugh Fisher 2004.

	Demonstrates a pixel shader that applies a screen
	door stencil effect within a window bounding sphere.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		24

#define cmdToggleShader	 1
#define cmdRotate	 2
#define cmdAnimate	 3
#define cmdExit		99

typedef struct {
	float x, y, r;
	float dx, dy;
	} Bound;

/* This is a demo, so use same color for diffuse and specular */
typedef struct {
	RGBA	color;
	float	shininess;
	} Material;

/* Global ambient lighting level instead of per-source */
static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };

static int	Menu;
static int	WinWidth, WinHeight;

static int	UsePixelProg, Animate;
static float	Rotation, CurrRotate;
static Bound	Lense;

static GLUquadricObj *	LitQuad;
static GLUquadricObj *  WireQuad;
static Material		Surface;
static GLfloat		LightPos[4];
static GLuint		Shader;

static char * ShaderSource =
/* Mesh lense, ARB version */
"!!ARBfp1.0						\n\
# Setup							\n\
ATTRIB  fPos	= fragment.position;			\n\
PARAM   lense	= program.env[1]; # x y square(r) 0	\n\
PARAM   ZERO	= { 0, 0, 0, 0 };			\n\
PARAM   NEG	= { -1, -1, -1, -1 };			\n\
PARAM   CONST	= { 0.25, 4.0, 0, 0 };			\n\
TEMP	dist, flag, coord;				\n\
MOV  flag, ZERO;					\n\
## Test if pixel inside lense area			\n\
# Distance from fragment to lense			\n\
SUB  dist, fragment.position, lense;			\n\
MUL  dist.z, dist.x, dist.x;	# sqr(x1 - x2)		\n\
MAD  dist.z, dist.y, dist.y, dist.z; # + sqr(y1 - y2)	\n\
# Is square(distance) < square(r) ?			\n\
SLT  flag.z, dist.z, lense.z;	# if <, z = 1		\n\
## Test if pixel x + y is divisible by N		\n\
ADD  coord.x, fPos.x, fPos.y;				\n\
MUL  coord.z, coord.x, CONST.x;	# z = sum / N		\n\
FLR  coord.z, coord.z;		# z = trunc(sum/N)	\n\
MUL  coord.z, coord.z, CONST.y; # z = N * trunc(sum/2)	\n\
SLT  flag.y, coord.z, coord.x;	# Divisible if x = z	\n\
# AND into distance test				\n\
MUL  flag.z, flag.z, flag.y;				\n\
## Skip pixel if inside and odd				\n\
MUL  flag.z, flag.z, NEG.z;	# Kill tests for -1	\n\
KIL  flag;						\n\
# else pass unchanged					\n\
MOV  result.color, fragment.color;			\n\
END\n\
";


/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
}

static void positionLight ()
{
  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  y = -4;
  n = 8;
  
  glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
  glDisable(GL_LIGHTING);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  /* Checkerboard ground plane */
  glColor3f(0, 1, 0);
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
  glPopAttrib();
}

static void apply (Material * mat)
{
  glColor4fv(mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat->color);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, mat->shininess);
}

static void drawShapes (float radius)
{
  int i;
  apply(&Surface);
  
  for (i = 0; i < 3; i++) {
    glPushMatrix();
      glRotatef(-i * 120, 0, 1, 0);
      glTranslatef(0, 0, radius);
      switch (i) {
        case 0:
    	  glutSolidTeapot(1.25);
	  break;
        case 1:
	  glTranslatef(0, 2, 0);
	  glRotatef(90, 1, 0, 0);
          gluCylinder(LitQuad, 0.5, 0.5, 4, 32, 16);
	  break;
	case 2:
    	  gluSphere(LitQuad, 1.25, 16, 16);
	  break;
	default:
	  break;
      }
    glPopMatrix();
  }
}

static void drawFrame ()
{
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  
  /* Pixel mode */
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, WinWidth, 0, WinHeight);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  
  glColor3f(1, 0, 0);
  glTranslatef(Lense.x, Lense.y, 0);
  gluDisk(WireQuad, 0, Lense.r, 32, 1);
  
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  
  glPopAttrib();
}

static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  lookFrom(0, 2, 12);
  positionLight();
  
  drawGround();
  
  if (UsePixelProg) {
    glEnable(GL_FRAGMENT_PROGRAM_ARB);
    glProgramEnvParameter4fARB(GL_FRAGMENT_PROGRAM_ARB, 1,
    			Lense.x, Lense.y, Lense.r * Lense.r, 0);
  }
  glPushMatrix();
    glRotatef(CurrRotate, 0, 1, 0);
    drawShapes(5);
  glPopMatrix();
 
  glDisable(GL_FRAGMENT_PROGRAM_ARB);
  
  drawFrame();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
  WinWidth  = width;
  WinHeight = height;
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdToggleShader:
      UsePixelProg = ! UsePixelProg;
      if (UsePixelProg)
        printf("Shader\n");
      else
        printf("Standard OpenGL\n");
      break;
    case cmdRotate:
      Rotation += 90;
      if (Rotation >= 360)
        Rotation = 0;
      break;
    case cmdAnimate:
      Animate = ! Animate;
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
  glutPostRedisplay();
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleShader);
  else if (key == 'R')
    menuChoice(cmdRotate);
  else if (key == ' ')
    menuChoice(cmdAnimate);
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0.75, 0.75, 0); 
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  
  glShadeModel(GL_SMOOTH);
  glEnable(GL_NORMALIZE);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, White);
}

static void initQuads ()
{
  LitQuad = gluNewQuadric();
  FailNull(LitQuad, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(LitQuad, GLU_FILL);
  gluQuadricOrientation(LitQuad, GLU_OUTSIDE);
  gluQuadricNormals(LitQuad, GLU_SMOOTH);
  gluQuadricTexture(LitQuad, GL_FALSE);
  
  WireQuad = gluNewQuadric();
  FailNull(WireQuad, "Cannot allocate quadric object");
  gluQuadricDrawStyle(WireQuad, GLU_SILHOUETTE);
  
  SetColor(Surface.color, 0.8, 0.4, 0);
  Surface.shininess = 10;
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

static void initShader ()
{
  int err;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_fragment_program"))
    Fail("GL_ARB_fragment_program not available on this machine");
  
  glEnable(GL_FRAGMENT_PROGRAM_ARB);
  glGenProgramsARB(1, &Shader);
  
  glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, Shader);
  glProgramStringARB(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			strlen(ShaderSource), ShaderSource);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0) {
    getErrorLine(ShaderSource, err, msg, sizeof(msg));
    printf("Error in shader : %s\n",
      	      glGetString(GL_PROGRAM_ERROR_STRING_ARB));
    printf("Source line:    %s\n", msg);
  }
}

static void initApp ()
{
  initGraphics();
  initQuads();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  LightPos[0] = 0;
  LightPos[1] = 1;
  LightPos[2] = 4;
  LightPos[3] = 0;
   
  Animate    = TRUE;
  Rotation   = 0;
  CurrRotate = 0;
    
  Lense.x = 200;
  Lense.y = 300;
  Lense.r = 64;
  Lense.dx = 2;
  Lense.dy = 2;
  
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Lense Shader", cmdToggleShader);
  glutAddMenuEntry("Rotate", cmdRotate);
  glutAddMenuEntry("Animate", cmdAnimate);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  /* Scene spin */
  if (CurrRotate < Rotation || (CurrRotate > 180 && Rotation < 180)) {
    CurrRotate += 2;
    if (CurrRotate > 360)
      CurrRotate = 0;
  }
  /* Lense */
  if (Animate) {
    Lense.x += Lense.dx;
    if (Lense.x + Lense.r >= WinWidth || Lense.x - Lense.r <= 0)
      Lense.dx *= -1;
    Lense.y += Lense.dy;
    if (Lense.y + Lense.r >= WinHeight || Lense.y - Lense.r <= 0)
      Lense.dy *= -1;
  }
  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("Lense fragment shader");

  initApp();
  
  printf("\n");
  printf("V toggles shader\n");
  printf("R key rotates scene\n");
  printf("space bar toggles lense movement\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
