
/*	Hugh Fisher 2003.

	Demonstrates vertical fogging instead of the usual
	distance, inspired by an underwater scene.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		24

#define cmdToggleShader	 1
#define cmdExit		99

/* Global ambient lighting level instead of per-source */
static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };

static int	Menu;

static int	UseVertProg, Animate;
static float	ViewRot;

#define NShapes	3
struct {
	float height;
	float step;
	} Bounce[NShapes];
	
/* This is a demo, so use same color for diffuse and specular */
typedef struct {
	RGBA	color;
	float	shininess;
	} Material;

static GLUquadricObj *	QState;
static Material		Surface;
static GLfloat		LightPos[4];
static GLuint		Shader;

static char * ShaderSource =
/* Shader for standard OpenGL lighting model with one distant light
   source, taken from ARB_vertex_program extension spec. Also uses
   Y instead of Z as basis for fog value. */
"!!ARBvp1.0					\n\
# Setup						\n\
ATTRIB	vPos	= vertex.position;		\n\
ATTRIB	vNorm	= vertex.normal;		\n\
ATTRIB	vCol	= vertex.color;			\n\
OUTPUT	oPos	= result.position;		\n\
OUTPUT	oCol	= result.color;			\n\
OUTPUT  oFog	= result.fogcoord;		\n\
PARAM	CTM[4]	= { state.matrix.mvp };		\n\
PARAM	IMV[4]	= { state.matrix.modelview.invtrans };	\n\
PARAM	lVec	= program.env[1];		\n\
PARAM   fogScale= program.env[2];		\n\
PARAM	lHalf	= state.light[0].half;		\n\
TEMP	vert, eyeNorm, coeff, shade, vFog;	\n\
PARAM	ONE  = { 1, 1, 1, 1 };			\n\
PARAM	HALF = { 0.5, 0.5, 0.5, 0.5 };		\n\
PARAM	NEG  = { -1, -1, -1, -1 };		\n\
## Standard transform				\n\
DP4  vert.x, CTM[0], vPos;			\n\
DP4  vert.y, CTM[1], vPos;			\n\
DP4  vert.z, CTM[2], vPos;			\n\
DP4  vert.w, CTM[3], vPos;			\n\
MOV  oPos, vert;				\n\
## Eye space surface normal			\n\
DP3  eyeNorm.x, IMV[0], vNorm;			\n\
DP3  eyeNorm.y, IMV[1], vNorm; 			\n\
DP3  eyeNorm.z, IMV[2], vNorm;			\n\
# and normalise					\n\
DP3  eyeNorm.w, eyeNorm, eyeNorm;		\n\
RSQ  eyeNorm.w, eyeNorm.w;			\n\
MUL  eyeNorm, eyeNorm, eyeNorm.w;		\n\
## Lighting					\n\
## Ambient: no setup required			\n\
MUL  shade, state.lightmodel.ambient, vCol;	\n\
## Diffuse: light dot normal			\n\
DP3  coeff.x, lVec, eyeNorm;			\n\
## Specular					\n\
DP3  coeff.y, lHalf, eyeNorm;			\n\
MOV  coeff.w, state.material.shininess.x;	\n\
# Eval coefficients and sum			\n\
LIT  coeff, coeff;				\n\
MAD  shade, coeff.y, vCol, shade;		\n\
MAD  shade, coeff.z, vCol, shade;		\n\
MOV  oCol, shade;				\n\
## Fog: use Y coord				\n\
MOV  vFog, vert.yyyw;				\n\
RCP  vFog.w, vFog.w;				\n\
MUL  vFog, vFog, vFog.w;			\n\
# Flip so downwards is further away		\n\
MUL  vFog, vFog, NEG;				\n\
# Adjust from -1..+1 to 0..1			\n\
ADD  vFog, vFog, ONE;				\n\
MUL  vFog, vFog, HALF;				\n\
# And map to user fog range			\n\
MUL  vFog, vFog, fogScale;			\n\
# Use maximum of Y and Z for final fog		\n\
MAX  vFog.x, vFog.x, vert.z;			\n\
MOV  oFog, vFog;				\n\
END\n\
";

/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
}

static void positionLight ()
{
  GLfloat vLight[4];

  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
  if (UseVertProg) {
    /* Pass light vector to vertex shader. Not strictly necessary
       as the position is part of the program state, but this way
       the vertex shader doesn't have to normalize it each time. */
    glGetLightfv(GL_LIGHT0, GL_POSITION, vLight);
    NormVec(vLight);
    glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 1, vLight);
  }
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  y = -4;
  n = 8;
  
  glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  /* Checkerboard ground plane */
  glColor3f(0, 1, 0);
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
  glPopAttrib();
}

static void apply (Material * mat)
{
  glColor4fv(mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat->color);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, mat->shininess);
}

static void drawShapes (float radius)
{
  int i;
  apply(&Surface);
  
  for (i = 0; i < NShapes; i++) {
    glPushMatrix();
      glRotatef(-i * 120, 0, 1, 0);
      glTranslatef(0, (Bounce[i].height * 4.0) - 2.0, radius);
      switch (i) {
        case 0:
    	  glutSolidTeapot(1.25);
	  break;
        case 1:
	  glTranslatef(0, 2, 0);
	  glRotatef(90, 1, 0, 0);
          gluCylinder(QState, 0.5, 0.5, 4, 32, 16);
	  break;
	case 2:
    	  gluSphere(QState, 1.25, 16, 16);
	  break;
	default:
	  break;
      }
    glPopMatrix();
  }
}


static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  lookFrom(0, 2, 12);
  positionLight();
  
  drawGround();
  if (UseVertProg) {
    glEnable(GL_VERTEX_PROGRAM_ARB);
    glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 2,
    			Far, 0, 0, 0);
  }
  
  glRotatef(ViewRot, 0, 1, 0);
  drawShapes(5);

  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
  
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdToggleShader:
      UseVertProg = ! UseVertProg;
      if (UseVertProg)
        printf("Vertex shader\n");
      else
        printf("Standard OpenGL\n");
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleShader);
  else if (key == ' ')
    Animate = ! Animate;
}


/****		Main control		****/


static GLfloat Sea[] = { 0, 0.75, 0.75, 0 };

static void initGraphics ()
{
  glClearColor(0, 0.75, 0.75, 0); 
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  
  glShadeModel(GL_SMOOTH);
  glEnable(GL_NORMALIZE);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, White);
  
  glEnable(GL_FOG);
  glFogi(GL_FOG_MODE, GL_LINEAR);
  glFogf(GL_FOG_START, Far/3.0);
  glFogf(GL_FOG_END, Far);
  glFogfv(GL_FOG_COLOR, Sea);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_FILL);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
  gluQuadricTexture(QState, GL_FALSE);
  
  SetColor(Surface.color, 0.8, 0.4, 0);
  Surface.shininess = 10;
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

static void initShader ()
{
  int  err;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
  
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  glGenProgramsARB(1, &Shader);
  
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader);
  glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			strlen(ShaderSource), ShaderSource);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0) {
    getErrorLine(ShaderSource, err, msg, sizeof(msg));
    printf("Error in shader %s\n",
      	      glGetString(GL_PROGRAM_ERROR_STRING_ARB));
    printf("Source line:    %s\n", msg);
  }
  glBindProgramARB(GL_VERTEX_PROGRAM_NV, Shader);
  				
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

static void initApp ()
{
  int i;
  
  initGraphics();
  initSphere();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  LightPos[0] = 0;
  LightPos[1] = 1;
  LightPos[2] = 4;
  LightPos[3] = 0;
   
  ViewRot = 0;
  Animate = TRUE;
  for (i = 0; i < NShapes; i++) {
    Bounce[i].height = i * 0.25;
    Bounce[i].step   = 0.006;
  }
    
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Vertex shader", cmdToggleShader);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  int i;
  
  if (! Animate)
    return;
    
  /* Scene spin */
  ViewRot += 0.1;
  if (ViewRot > 360)
    ViewRot = 0;
    
  /* Object bounce */
  for (i = 0; i < NShapes; i++) {
    if (Bounce[i].height < 0) {
      Bounce[i].height = 0;
      Bounce[i].step *= -1;
    } else if (Bounce[i].height > 1) {
      Bounce[i].height = 1;
      Bounce[i].step *= -1;
    } else
      Bounce[i].height += Bounce[i].step;
  }

  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("ARB vertical fog");

  initApp();
  
  printf("\n");
  printf("V toggles vertex shader\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
