
/*	Hugh Fisher 2003.

	Vertex program demonstrating simple free form
	deformation on arbitrary geometries. The shader
	checks if vertices are within a bounding frame,
	and if so changes the color and moves the coords.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		20

#define cmdRotate	 1
#define cmdToggleFrame	 2
#define cmdExit		99

static int	Menu;

/* Scene management */
static float	Rotation;
#define Radius	8
static int	ShowFrame;

/* Animation control */
static float	CurrRotate;
static float	Height;
static float	BounceStep;

static GLUquadricObj *	QState;
#define NumPoints	64
static GLfloat Particles[NumPoints * 3];


/* The frame that defines the volume of effect */
#define BBX	1.25
#define BBY	0.25
#define BBZ	1.25

/* Visual outline of the frame */
static GLfloat CubeVerts[8][3] = {
  { -BBX, -BBY, -BBZ },
  { -BBX, -BBY,  BBZ },
  {  BBX, -BBY,  BBZ },
  {  BBX, -BBY, -BBZ },
  { -BBX,  BBY, -BBZ },
  { -BBX,  BBY,  BBZ },
  {  BBX,  BBY,  BBZ },
  {  BBX,  BBY, -BBZ },
};

static GLint CubeEdges[] = {
  0, 1, 1, 2, 2, 3, 3, 0,
  4, 5, 5, 6, 6, 7, 7, 4,
  0, 4, 1, 5, 2, 6, 3, 7
};

#define NProgs		1
static GLuint		Shader[NProgs];

static char * ShaderSrc[NProgs] = {
/* Free form deformation shader */
"!!VP2.0						\n\
# Setup							\n\
# c[0-3]	= CTM					\n\
# c[4-7]	= inverse MV				\n\
# c[8-11]	= MV					\n\
# c[12-15]	= Proj					\n\
# c[58]		= eye pos				\n\
# c[59]		= -1, 0, 1, 0.5				\n\
# c[60]		= FFD position				\n\
# c[61]		= FFD size				\n\
# c[62]		= deformation x, y, z			\n\
## Transform to world space				\n\
DP4  R1.x, c[8], v[OPOS];				\n\
DP4  R1.y, c[9], v[OPOS];				\n\
DP4  R1.z, c[10], v[OPOS];				\n\
DP4  R1.w, c[11], v[OPOS];				\n\
# No lighting						\n\
MOV  o[COL0], v[COL0];					\n\
## Test if inside FFD box				\n\
MOV  R0, c[60];		# FFD pos			\n\
SUB  R0, R0, c[61];	# lower left corner		\n\
SLT  R0, R1, R0;	# Test if vert < corner		\n\
DP3C R0.x, R0, R0;	# Sum xyz tests			\n\
BRA  outside (GT.x);	# Skip rest if any <		\n\
MOV  R0, c[60];		# FFD pos			\n\
ADD  R0, R0, c[61];	# Upper right			\n\
SGT  R0, R1, R0;	# Test if vert > corner		\n\
DP3C R0.x, R0, R0;	# Sum xyz tests			\n\
BRA  outside (GT.x);	# Skip rest if any >		\n\
## Deform						\n\
MOV  o[COL0], c[59].zyyz; # Recolor red			\n\
SUB  R0, c[60], R1;	# dist = FFD pos - vert		\n\
MUL  R0, R0, c[62];	# apply scale			\n\
ADD  R1, R1, R0;	# and add to vert		\n\
## Final transform to eye space				\n\
outside:						\n\
DP4  o[HPOS].x, c[12], R1;				\n\
DP4  o[HPOS].y, c[13], R1;				\n\
DP4  o[HPOS].z, c[14], R1;				\n\
DP4  o[HPOS].w, c[15], R1;				\n\
END\n\
"
};


/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 58, x, y, z, 0);
}

static void drawFrame ()
{
  GLfloat mv[16];
  
  /* Don't deform ourself! */
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 61, 0, 0, 0, 0);

  glColor3f(0, 1, 1);
  glEnableClientState(GL_VERTEX_ARRAY);
  glPushMatrix();
    glTranslatef(0, Height, Radius);
    if (ShowFrame) {
      glVertexPointer(3, GL_FLOAT, 0, CubeVerts);
      glDrawElements(GL_LINES, 24, GL_UNSIGNED_INT, CubeEdges);
    }
    /* Find out where we just drew the box */
    glGetFloatv(GL_MODELVIEW_MATRIX, mv);
  glPopMatrix();
  glDisableClientState(GL_VERTEX_ARRAY);
  
  /* Pass current bounding box world position and dimensions to shader.
     Position is equal to translation in modelview matrix. */
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 60, mv[12], mv[13], mv[14], 0);
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 61, BBX, BBY, BBZ, 0);
  
  /* Current deformation factors */
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 62, 0.25, 0, 0.25, 0);
}

static void drawShapes ()
{
  int i;
  
  glColor3f(1, 1, 1);
  glPushMatrix();
  glRotatef(CurrRotate, 0, 1, 0);
  
  for (i = 0; i < 4; i++) {
    glPushMatrix();
      glRotatef(-i * 90, 0, 1, 0);
      glTranslatef(0, 0, Radius);
      switch (i) {
        case 0:
	  /* Nothing */
	  break;
	case 1:
	  glEnableClientState(GL_VERTEX_ARRAY);
	  glVertexPointer(3, GL_FLOAT, 0, Particles);
	  glDrawArrays(GL_POINTS, 0, NumPoints);
	  glDisableClientState(GL_VERTEX_ARRAY);
	  break;
        case 2:
	  glTranslatef(0, 2, 0);
	  glRotatef(90, 1, 0, 0);
          gluCylinder(QState, 0.5, 0.5, 4, 32, 16);
	  break;
	case 3:
	  glScalef(1, 3, 1);
	  glutWireTeapot(1);
	  break;
	default:
	  break;
      }
    glPopMatrix();
  }
  glPopMatrix();
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  y = -4;
  n = 8;
  
  /* Checkerboard ground plane */
  glColor3f(0.25, 0.25, 0.25);
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
}

static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  lookFrom(0, 4, 16);
  drawGround();
  drawFrame();
  drawShapes();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdRotate:
      Rotation += 90;
      if (Rotation >= 360)
        Rotation = 0;
      break;
    case cmdToggleFrame:
      ShowFrame = ! ShowFrame;
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'R')
    menuChoice(cmdRotate);
  else if (key == 'F')
    menuChoice(cmdToggleFrame);
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_POINT_SMOOTH);
  glPointSize(4);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_FILL);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
  gluQuadricTexture(QState, GL_FALSE);
}

static float rnum (float max)
{
  return (((float)rand() / (float)RAND_MAX) * max) - (max/2.0);
}

static void initParticles ()
{
  int i, idx;
  
  srand(time(NULL));
  
  idx = 0;
  for (i = 0; i < NumPoints; i++) {
    /* Random position */
    Particles[idx] = rnum(1); idx ++;
    Particles[idx] = rnum(4); idx ++;
    Particles[idx] = rnum(1); idx ++;
  }
  
}

static void initShader ()
{
  int  err, i;
  
  if (! glutExtensionSupported("GL_NV_vertex_program2"))
    Fail("GL_NV_vertex_program2 not available on this machine");
  
  glEnable(GL_VERTEX_PROGRAM_NV);
  
  glGenProgramsNV(NProgs, Shader);
  
  for (i = 0; i < NProgs; i++) {
    glLoadProgramNV(GL_VERTEX_PROGRAM_NV, Shader[i],
  			  strlen(ShaderSrc[i]), ShaderSrc[i]);
    glGetIntegerv(GL_PROGRAM_ERROR_POSITION_NV, &err);
    if (err >= 0) {
      printf("Error %d in shader #%d\n", err, i);
    }
  }
  
  glBindProgramNV(GL_VERTEX_PROGRAM_NV, Shader[0]);
  				
  /* Make matrices available to shader in constant registers */
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0,
  		GL_MODELVIEW_PROJECTION_NV, GL_IDENTITY_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4,
  		GL_MODELVIEW, GL_INVERSE_TRANSPOSE_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 8,
  		GL_MODELVIEW, GL_IDENTITY_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 12,
  		GL_PROJECTION, GL_IDENTITY_NV);

  /* Useful values. */
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 59, -1, 0, 1, 0.5);
}

static void initApp ()
{
  initGraphics();
  initSphere();
  initParticles();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  Height     = 1;
  BounceStep = 0.02;
  ShowFrame  = TRUE;
  
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Rotate", cmdRotate);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  /* Smooth scene rotation */
  if (CurrRotate < Rotation || (CurrRotate > 180 && Rotation < 180)) {
    CurrRotate += 2;
    if (CurrRotate > 360)
      CurrRotate = 0;
  }
  
  /* Bounce the ball up or down */
  if (Height < -2) {
    Height = -2;
    BounceStep = - BounceStep;
  } else if (Height > 3) {
    Height = 3;
    BounceStep = - BounceStep;
  } else
    Height += BounceStep;

  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("Vertex FFD");

  initApp();
  
  printf("\n");
  printf("R rotates objects\n");
  printf("F toggles frame\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
