
/*	Hugh Fisher 2003.

	ARB program demonstrating simple free form
	deformation on arbitrary geometries. The shader
	checks if vertices are within a bounding frame,
	and if so changes the color and moves the coords.
	
	As you can see by comparing the code to the NV
	equivalent, doing this kind of decision logic
	without any branch instructions is a complete
	pain in the butt but I had to do it once just
	to see if it was possible.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		20

#define cmdRotate	 1
#define cmdToggleFrame	 2
#define cmdAnimate	 3
#define cmdExit		99

static int	Menu;

/* Scene management */
static float	Rotation;
#define Radius	8
static int	ShowFrame;
static RGBA	Background = { 0, 0, 0, 1 };

/* Animation control */
static int	Animate;
static float	CurrRotate;
static float	Height;
static float	BounceStep;

static GLUquadricObj *	QState;
#define NumPoints	64
static GLfloat Particles[NumPoints * 3];


/* The frame that defines the volume of effect */
#define BBX	1.25
#define BBY	0.25
#define BBZ	1.25

/* Visual outline of the frame */
static GLfloat CubeVerts[8][3] = {
  { -BBX, -BBY, -BBZ },
  { -BBX, -BBY,  BBZ },
  {  BBX, -BBY,  BBZ },
  {  BBX, -BBY, -BBZ },
  { -BBX,  BBY, -BBZ },
  { -BBX,  BBY,  BBZ },
  {  BBX,  BBY,  BBZ },
  {  BBX,  BBY, -BBZ },
};

static GLint CubeEdges[] = {
  0, 1, 1, 2, 2, 3, 3, 0,
  4, 5, 5, 6, 6, 7, 7, 4,
  0, 4, 1, 5, 2, 6, 3, 7
};

#define NProgs		1
static GLuint		Shader[NProgs];

static char * ShaderSrc[NProgs] = {
/* Free form deformation shader, ARB version */
"!!ARBvp1.0						\n\
# Setup							\n\
ATTRIB	vPos	= vertex.position;			\n\
ATTRIB	vCol	= vertex.color;				\n\
OUTPUT	oPos	= result.position;			\n\
OUTPUT	oCol	= result.color;				\n\
PARAM	PRM[4]	= { state.matrix.projection };		\n\
PARAM	MVM[4]	= { state.matrix.modelview };		\n\
PARAM	ffdPos	= program.env[1];			\n\
PARAM	ffdSize	= program.env[2];			\n\
PARAM	ffdScale= program.env[3];			\n\
TEMP	wPos, ll, ur, mask, notMask, dist, flags, rgba;	\n\
PARAM	ZERO = { 0, 0, 0, 0 };				\n\
PARAM	ONE  = { 1, 1, 1, 1 };				\n\
PARAM	RED  	= { 1, 0, 0, 1 };			\n\
## Transform to world space				\n\
DP4  wPos.x, MVM[0], vPos;				\n\
DP4  wPos.y, MVM[1], vPos;				\n\
DP4  wPos.z, MVM[2], vPos;				\n\
DP4  wPos.w, MVM[3], vPos;				\n\
# No lighting						\n\
MOV  oCol, vCol;					\n\
## Test if inside FFD box				\n\
## Because we don't have branches in ARB, we want to	\n\
## set the mask register to all 0 if the vertex is	\n\
## outside the FFD box, all 1 if inside			\n\
MOV  mask, ZERO;					\n\
# First test, >= lower left corner?			\n\
MOV  ll, ffdPos;		# FFD pos		\n\
SUB  ll, ll, ffdSize;		# lower left corner	\n\
# Set flags to all 1 if all >=, otherwise 0		\n\
SLT  flags, wPos, ll;		# Test components	\n\
DP3  flags, flags, flags;	# 0 if all components 0	\n\
MIN  flags, flags, ONE;		# and 1 otherwise	\n\
SUB  flags, flags, ONE;		# Logical NOT		\n\
MOV  mask, flags;					\n\
# Second test, <= upper right?				\n\
MOV  ur, ffdPos;		# FFD pos		\n\
ADD  ur, ur, ffdSize;		# Upper right		\n\
SGE  flags, wPos, ur;		# Test components	\n\
DP3  flags, flags, flags;	# 0 if all components 0	\n\
MIN  flags, flags, ONE;		# and 1 otherwise	\n\
SUB  flags, flags, ONE;		# Logical NOT		\n\
MUL  mask, mask, flags;		# Logical AND		\n\
## Change color? Would be easier if vertex had LRP	\n\
SUB  notMask, ONE, mask;				\n\
MUL  rgba, notMask, vCol;	# (mask - 1) * orig	\n\
MAD  rgba, mask, RED, rgba;	# + mask * red		\n\
MOV  oCol, rgba;					\n\
# Deform vertex?					\n\
SUB  dist, ffdPos, wPos;	# dist = FFD pos - vert	\n\
MUL  dist, dist, ffdScale;	# apply scale		\n\
MUL  dist, dist, mask;		# apply FFD box test	\n\
ADD  wPos, wPos, dist;		# and add to vert	\n\
## Final transform to eye space				\n\
DP4  oPos.x, PRM[0], wPos;				\n\
DP4  oPos.y, PRM[1], wPos;				\n\
DP4  oPos.z, PRM[2], wPos;				\n\
DP4  oPos.w, PRM[3], wPos;				\n\
END\n\
"
};


/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
}

static void drawFrame ()
{
  GLfloat mv[16];
  
  /* Don't deform ourself! */
  glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 2, 0, 0, 0, 0);

  glEnableClientState(GL_VERTEX_ARRAY);
  glColor3f(0, 1, 1);
  glPushMatrix();
    glTranslatef(0, Height, Radius);
    if (ShowFrame) {
      glVertexPointer(3, GL_FLOAT, 0, CubeVerts);
      glDrawElements(GL_LINES, 24, GL_UNSIGNED_INT, CubeEdges);
    }
    /* Find out where we just drew the box */
    glGetFloatv(GL_MODELVIEW_MATRIX, mv);
  glPopMatrix();
  glDisableClientState(GL_VERTEX_ARRAY);
  
  /* Pass current bounding box world position and dimensions to shader.
     Position is equal to translation in modelview matrix. */
  glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 1, mv[12], mv[13], mv[14], 0);
  glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 2, BBX, BBY, BBZ, 0);
  
  /* Current deformation factors */
  glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 3, 0.25, 0, 0.25, 0);
  glProgramEnvParameter4fARB(GL_VERTEX_PROGRAM_ARB, 3, 0.25, 0, 0.25, 0);
}

static void drawShapes ()
{
  int i;
  
  glColor3f(0, 1, 0);
  glPushMatrix();
  glRotatef(CurrRotate, 0, 1, 0);
  
  for (i = 0; i < 4; i++) {
    glPushMatrix();
      glRotatef(-i * 90, 0, 1, 0);
      glTranslatef(0, 0, Radius);
      switch (i) {
        case 0:
	  /* Nothing */
	  break;
	case 1:
	  glEnableClientState(GL_VERTEX_ARRAY);
	  glVertexPointer(3, GL_FLOAT, 0, Particles);
	  glDrawArrays(GL_POINTS, 0, NumPoints);
	  glDisableClientState(GL_VERTEX_ARRAY);
	  break;
        case 2:
	  glTranslatef(0, 2, 0);
	  glRotatef(90, 1, 0, 0);
          gluCylinder(QState, 0.5, 0.5, 4, 32, 16);
	  break;
	case 3:
	  glScalef(1, 3, 1);
	  glutWireTeapot(1);
	  break;
	default:
	  break;
      }
    glPopMatrix();
  }
  glPopMatrix();
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  y = -4;
  n = 8;
  
  /* Checkerboard ground plane */
  glColor3f(0.25, 0.25, 0.25);
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
}

static void display ()
{
  glClearColor(Background[0], Background[1], Background[2], Background[3]);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  lookFrom(0, 2, 13);
  drawGround();
  drawFrame();
  drawShapes();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdRotate:
      Rotation += 90;
      if (Rotation >= 360)
        Rotation = 0;
      break;
    case cmdToggleFrame:
      ShowFrame = ! ShowFrame;
      break;
    case cmdAnimate:
      Animate = ! Animate;
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'R')
    menuChoice(cmdRotate);
  else if (key == 'F')
    menuChoice(cmdToggleFrame);
  else if (key == ' ')
    menuChoice(cmdAnimate);
  else if (key == 'W')
    SetColor(Background, 1, 1, 1);
  else if (key == 'B')
    SetColor(Background, 0, 0, 0);
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_POINT_SMOOTH);
  glPointSize(4);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_FILL);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
  gluQuadricTexture(QState, GL_FALSE);
}

static float rnum (float max)
{
  return (((float)rand() / (float)RAND_MAX) * max) - (max/2.0);
}

static void initParticles ()
{
  int i, idx;
  
  srand(time(NULL));
  
  idx = 0;
  for (i = 0; i < NumPoints; i++) {
    /* Random position */
    Particles[idx] = rnum(1); idx ++;
    Particles[idx] = rnum(4); idx ++;
    Particles[idx] = rnum(1); idx ++;
  }
  
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

static void initShader ()
{
  int  err, i;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
  
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  glGenProgramsARB(NProgs, Shader);
    
  for (i = 0; i < NProgs; i++) {
    glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader[i]);
    glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			  strlen(ShaderSrc[i]), ShaderSrc[i]);
    glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
    if (err >= 0) {
      getErrorLine(ShaderSrc[i], err, msg, sizeof(msg));
      printf("Error in shader #%d: %s\n", i,
      		glGetString(GL_PROGRAM_ERROR_STRING_ARB));
      printf("Source line:    %s\n", msg);
    }
  }  				
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader[0]);
}

static void initApp ()
{
  initGraphics();
  initSphere();
  initParticles();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  Height     = 1;
  BounceStep = 0.02;
  ShowFrame  = TRUE;
  Animate    = TRUE;
  
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Rotate", cmdRotate);
  glutAddMenuEntry("Frame", cmdToggleFrame);
  glutAddMenuEntry("Animate", cmdAnimate);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  /* Smooth scene rotation */
  if (CurrRotate < Rotation || (CurrRotate > 180 && Rotation < 180)) {
    CurrRotate += 2;
    if (CurrRotate > 360)
      CurrRotate = 0;
  }
  
  /* Bounce the ball up or down */
  if (Animate) {
    if (Height < -2) {
      Height = -2;
      BounceStep = - BounceStep;
    } else if (Height > 3) {
      Height = 3;
      BounceStep = - BounceStep;
    } else
      Height += BounceStep;
  }
  
  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("ARB Vertex FFD");

  initApp();
  
  printf("\n");
  printf("R rotates objects\n");
  printf("F toggles frame\n");
  printf("space starts/stops animation\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
