
/*	Hugh Fisher 2003.

	Demonstrates two simple vertex programs:
	* Standard OpenGL transform and lighting for a single
	  directional light source. If you want to extend OpenGL,
	  you really ought to know how it works first :-)
	* An infinite loop, to see how the graphics card reacts
	  to a faulty program.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		24

#define cmdToggleShader	 1
#define cmdLoadBomb	 2
#define cmdExit		99

/* Global ambient lighting level instead of per-source */
static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };

/* Infinite viewer vector */
static GLfloat IVP[4] = { 0, 0, 1, 0 };

static int	Menu;

static int	UseVertProg;

/* Animation control */
static float	Height;
static float	BounceStep;

/* This is a demo, so use same color for diffuse and specular */
typedef struct {
	RGBA	color;
	float	shininess;
	} Material;

static GLUquadricObj *	QState;
static Material		Surface;
static GLfloat		LightPos[4];

static long		TimeBase;

#define NProgs		2
static GLuint		Shader[NProgs];

static char * ShaderSrc[NProgs] = {
/* Shader for standard OpenGL lighting model with one distant light
   source. Based on nVidia "Fixed Function Pipeline" tutorial. */
"!!VP2.0			\n\
# Setup				\n\
# c[0-3]	= CTM		\n\
# c[4-7]	= inverse MV	\n\
# c[8-11]	= MV		\n\
# c[12-15]	= Proj		\n\
# c[40]	= ambient		\n\
# c[41]	= light pos		\n\
# c[42] = specular half vector	\n\
# c[43]	= shininess		\n\
# c[58]	= eye pos		\n\
# c[59]	= -1, 0, 1, 0.5		\n\
## Standard transform		\n\
DP4  o[HPOS].x, c[0], v[OPOS];	\n\
DP4  o[HPOS].y, c[1], v[OPOS];	\n\
DP4  o[HPOS].z, c[2], v[OPOS];	\n\
DP4  o[HPOS].w, c[3], v[OPOS];	\n\
## Eye space surface normal	\n\
DP3  R1.x, v[NRML], c[4];	\n\
DP3  R1.y, v[NRML], c[5]; 	\n\
DP3  R1.z, v[NRML], c[6];	\n\
# and normalise			\n\
DP3  R0.w, R1, R1;		\n\
RSQ  R0.w, R0.w;		\n\
MUL  R1, R1, R0.w;		\n\
## Lighting			\n\
## Ambient: no setup required	\n\
MUL  R6, c[40], v[COL0];	\n\
## Diffuse: light dot normal	\n\
DP3  R5.x, c[41], R1;		\n\
## Specular			\n\
DP3  R5.y, c[42], R1;		\n\
MOV  R5.w, c[43].x;		\n\
# And eval coefficients		\n\
LIT  R4, R5;			\n\
MUL  R7, v[COL0], R4.y;		\n\
MUL  R8, v[COL0], R4.z;		\n\
# Combine			\n\
ADD  R6, R6, R7;		\n\
ADD  R6, R6, R8;		\n\
MOV  o[COL0], R6;		\n\
END\n\
"
,
/* Infinite loop shader	*/
"!!VP2.0			\n\
# c[59]	= -1, 0, 1, 0.5		\n\
MOVC  R0.x, c[59].z;		\n\
start:				\n\
SUBC R0.x, R0.x, c[59].y;	\n\
BRA start (LE.x);		\n\
MOV   o[HPOS], v[OPOS];		\n\
MOV   o[COL0], v[COL0];		\n\
END\n\
"
};


/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
  if (UseVertProg)
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 58, x, y, z, 0);
}

static void positionLight ()
{
  GLfloat vLight[4];
  GLfloat half[4];

  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
  if (UseVertProg) {
    /* Important: the vertex program needs the actual light position
       after transformation into eye space, not world space pos */
    glGetLightfv(GL_LIGHT0, GL_POSITION, vLight);
    /* Specular lighting with directional light, infinite viewer */
    NormVec(vLight);
    glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, 41, vLight);
    AddVec(half, vLight, IVP);
    NormVec(half);
    glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, 42, half);
  }
}

static void apply (Material * mat)
{
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat->color);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, mat->shininess);
  if (UseVertProg) {
    glColor4fv(mat->color);
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 43, mat->shininess, 0, 0, 0);
  }
}

static void drawSphere ()
{
  apply(&Surface);
  glPushMatrix();
    glTranslatef(0, Height, 0);
    gluSphere(QState, 1, 32, 16);
  glPopMatrix();
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  apply(&Surface);
    
  y = -3;
  n = 6;
  
  /* Ground plane: need to tessellate for lighting */
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
}

static void drawScenery ()
{
  apply(&Surface);
  
  glPushMatrix();
    glTranslatef(-3, -2.5, 0);
    glutSolidTeapot(1);
  glPopMatrix();
  
  glPushMatrix();
    glTranslatef(0, -3, -3);
    glRotatef(-90, 1, 0, 0);
    gluCylinder(QState, 0.5, 0.5, 4, 8, 8);
  glPopMatrix();
}


static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  if (UseVertProg)
    glEnable(GL_VERTEX_PROGRAM_NV);
  
  lookFrom(0, 2, 8);
  positionLight();
  
  drawSphere();
  drawGround();
  drawScenery();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
  
  glDisable(GL_VERTEX_PROGRAM_NV);
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdToggleShader:
      UseVertProg = ! UseVertProg;
      if (UseVertProg)
        printf("Vertex shader\n");
      else
        printf("Standard OpenGL\n");
      break;
    case cmdLoadBomb:
      if (! UseVertProg)
        menuChoice(cmdToggleShader);
      glBindProgramNV(GL_VERTEX_PROGRAM_NV, Shader[1]);
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleShader);
  else if (key == 'B')
    menuChoice(cmdLoadBomb);
  else if (key == '-' && LightPos[0] > -4)
    LightPos[0] -= 0.5;
  else if (key == '=' && LightPos[0] < 4)
    LightPos[0] += 0.5;
  else if (key == 'M')
    Surface.shininess += 1;
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0); 
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  
  glShadeModel(GL_SMOOTH);
  glEnable(GL_NORMALIZE);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, White);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_FILL);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
  gluQuadricTexture(QState, GL_FALSE);
  
  SetColor(Surface.color, 0.4, 0.4, 0.8);
  Surface.shininess = 20;
}

static int microsecs ()
{
  struct timeval t;

  gettimeofday(&t, NULL);
  return (t.tv_sec - TimeBase) * 1000000 + t.tv_usec;
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}


static void initShader ()
{
  int  err, i;
  long start, finish;
  char msg[256];
  
  if (! glutExtensionSupported("GL_NV_vertex_program2"))
    Fail("GL_NV_vertex_program2 not available on this machine");
  
  glEnable(GL_VERTEX_PROGRAM_NV);
  
  glGenProgramsNV(NProgs, Shader);
  
  for (i = 0; i < NProgs; i++) {
    start = microsecs();
    glLoadProgramNV(GL_VERTEX_PROGRAM_NV, Shader[i],
  			  strlen(ShaderSrc[i]), ShaderSrc[i]);
    finish = microsecs();
    printf("Elapsed = 0.%06d\n", (int)(finish - start));
    glGetIntegerv(GL_PROGRAM_ERROR_POSITION_NV, &err);
    if (err >= 0) {
      getErrorLine(ShaderSrc[i], err, msg, sizeof(msg));
      printf("Error in shader #%d\n", i);
      printf("Source line:    %s\n", msg);
    }
  }
  
  glBindProgramNV(GL_VERTEX_PROGRAM_NV, Shader[0]);
  				
  /* Make matrices available to shader in constant registers */
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0,
  		GL_MODELVIEW_PROJECTION_NV, GL_IDENTITY_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4,
  		GL_MODELVIEW, GL_INVERSE_TRANSPOSE_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 8,
  		GL_MODELVIEW, GL_IDENTITY_NV);
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 12,
  		GL_PROJECTION, GL_IDENTITY_NV);

  /* Fixed ambient */
  glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, 40, Ambient);
  
  /* Useful values. Make sure GL_GLEXT_PROTOTYPES is defined,
     otherwise the compiler won't convert integer constants
     to the floating point required. */
  glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 59, -1, 0, 1, 0.5);
  
  glDisable(GL_VERTEX_PROGRAM_NV);
}

static void initApp ()
{
  TimeBase = time(NULL);
  
  initGraphics();
  initSphere();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  LightPos[0] = 0;
  LightPos[1] = 1;
  LightPos[2] = 4;
  LightPos[3] = 0;
   
  Height     = 1;
  BounceStep = 0.05;
    
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Vertex shader", cmdToggleShader);
  glutAddMenuEntry("Infinite loop", cmdLoadBomb);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  /* Bounce the ball up or down */
  if (Height < -2) {
    Height = -2;
    BounceStep = - BounceStep;
  } else if (Height > 3) {
    Height = 3;
    BounceStep = - BounceStep;
  } else
    Height += BounceStep;

  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("nVidia vertex program");

  initApp();
  
  printf("\n");
  printf("V toggles vertex shader\n");
  printf("B loads bomb shader\n");
  printf("- and = move light left or right\n");
  printf("M increases shininess\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
