
/*	Hugh Fisher 2003.

	Demonstrates simple ARB implementation of standard
	OpenGL transform and lighting for one directional light.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		24

#define cmdToggleShader	 1
#define cmdExit		99

/* Global ambient lighting level instead of per-source */
static RGBA Ambient = { 0.2, 0.2, 0.2, 1.0 };

static int	Menu;

static int	UseVertProg;

/* Animation control */
static float	Height;
static float	BounceStep;

/* This is a demo, so use same color for diffuse and specular */
typedef struct {
	RGBA	color;
	float	shininess;
	} Material;

static GLUquadricObj *	QState;
static Material		Surface;
static GLfloat		LightPos[4];

static long		TimeBase;

#define NProgs		1
static GLuint		Shader[NProgs];

static char * ShaderSrc[NProgs] = {
/* Shader for standard OpenGL lighting model with one distant light
   source. Taken from ARB_vertex_program extension spec. */
"!!ARBvp1.0					\n\
# Setup						\n\
ATTRIB	vPos	= vertex.position;		\n\
ATTRIB	vNorm	= vertex.normal;		\n\
ATTRIB	vCol	= vertex.color;			\n\
OUTPUT	oPos	= result.position;		\n\
OUTPUT	oCol	= result.color;			\n\
PARAM	CTM[4]	= { state.matrix.mvp };		\n\
PARAM	IMV[4]	= { state.matrix.modelview.invtrans };	\n\
PARAM	lVec	= program.env[1];		\n\
PARAM	lHalf	= state.light[0].half;		\n\
TEMP	eyeNorm, coeff, shade;			\n\
PARAM	red = { 1, 0, 0, 1 };			\n\
## Standard transform				\n\
DP4  oPos.x, CTM[0], vPos;			\n\
DP4  oPos.y, CTM[1], vPos;			\n\
DP4  oPos.z, CTM[2], vPos;			\n\
DP4  oPos.w, CTM[3], vPos;			\n\
## Eye space surface normal			\n\
DP3  eyeNorm.x, IMV[0], vNorm;			\n\
DP3  eyeNorm.y, IMV[1], vNorm; 			\n\
DP3  eyeNorm.z, IMV[2], vNorm;			\n\
# and normalise					\n\
DP3  eyeNorm.w, eyeNorm, eyeNorm;		\n\
RSQ  eyeNorm.w, eyeNorm.w;			\n\
MUL  eyeNorm, eyeNorm, eyeNorm.w;		\n\
## Lighting					\n\
## Ambient: no setup required			\n\
MUL  shade, state.lightmodel.ambient, vCol;	\n\
## Diffuse: light dot normal			\n\
DP3  coeff.x, lVec, eyeNorm;			\n\
## Specular					\n\
DP3  coeff.y, lHalf, eyeNorm;			\n\
MOV  coeff.w, state.material.shininess.x;	\n\
# Eval coefficients and sum			\n\
LIT  coeff, coeff;				\n\
MAD  shade, coeff.y, vCol, shade;		\n\
MAD  shade, coeff.z, vCol, shade;		\n\
MOV  oCol, shade;				\n\
END\n\
"
};

/****		Drawing the world		****/

static void lookFrom (float x, float y, float z)
{
  gluLookAt(x, y, z, 0, 0, 0, 0, 1, 0);
}

static void positionLight ()
{
  GLfloat vLight[4];

  glLightfv(GL_LIGHT0, GL_POSITION, LightPos);
  if (UseVertProg) {
    /* Pass light vector to vertex shader. Not strictly necessary
       as the position is part of the program state, but this way
       the vertex shader doesn't have to normalize it each time. */
    glGetLightfv(GL_LIGHT0, GL_POSITION, vLight);
    NormVec(vLight);
    glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 1, vLight);
  }
}

static void apply (Material * mat)
{
  glColor4fv(mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat->color);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat->color);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, mat->shininess);
}

static void drawSphere ()
{
  apply(&Surface);
  glPushMatrix();
    glTranslatef(0, Height, 0);
    gluSphere(QState, 1, 32, 16);
  glPopMatrix();
}

static void drawGround ()
{
  int   n, i, j;
  float y;
  
  apply(&Surface);
    
  y = -3;
  n = 6;
  
  /* Ground plane: need to tessellate for lighting */
  for (i = -n; i < n - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    glNormal3f(0, 1, 0);
    for (j = -n; j < n; j++) {
      glVertex3f(j, y, i);
      glVertex3f(j, y, i + 1);
    }
    glEnd();
  }
}

static void drawScenery ()
{
  apply(&Surface);
  
  glPushMatrix();
    glTranslatef(-3, -2.5, 0);
    glutSolidTeapot(1);
  glPopMatrix();
  
  glPushMatrix();
    glTranslatef(0, -3, -3);
    glRotatef(-90, 1, 0, 0);
    gluCylinder(QState, 0.5, 0.5, 4, 8, 8);
  glPopMatrix();
}


static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  if (UseVertProg)
    glEnable(GL_VERTEX_PROGRAM_ARB);
  
  lookFrom(0, 2, 8);
  positionLight();
  
  drawSphere();
  drawGround();
  drawScenery();
  
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
  
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdToggleShader:
      UseVertProg = ! UseVertProg;
      if (UseVertProg)
        printf("Vertex shader\n");
      else
        printf("Standard OpenGL\n");
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleShader);
  else if (key == '-' && LightPos[0] > -4)
    LightPos[0] -= 0.5;
  else if (key == '=' && LightPos[0] < 4)
    LightPos[0] += 0.5;
  else if (key == 'M')
    Surface.shininess += 1;
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0); 
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  
  glShadeModel(GL_SMOOTH);
  glEnable(GL_NORMALIZE);
  
  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, Ambient);
  glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_AMBIENT, Black);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, White);
  glLightfv(GL_LIGHT0, GL_SPECULAR, White);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_FILL);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
  gluQuadricTexture(QState, GL_FALSE);
  
  SetColor(Surface.color, 0.4, 0.4, 0.8);
  Surface.shininess = 20;
}

static int microsecs ()
{
  struct timeval t;

  gettimeofday(&t, NULL);
  return (t.tv_sec - TimeBase) * 1000000 + t.tv_usec;
}

static void getErrorLine (char * src, int err, char line[], int maxLine)
{
  int start, finish, n;
  
  start = err;
  while (start >= 0 && isprint(src[start]))
    start --;
  start ++;
  finish = err;
  while (isprint(src[finish]))
    finish ++;
  n = finish - start;
  if (n >= maxLine)
    n = maxLine;
  strncpy(line, src + start, n);
  line[n] = 0;
}

static void initShader ()
{
  int  err, i;
  long start, finish;
  char msg[256];
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
  
  glEnable(GL_VERTEX_PROGRAM_ARB);
  
  glGenProgramsARB(NProgs, Shader);
  
  for (i = 0; i < NProgs; i++) {
    start = microsecs();
    /* Different from nVidia: bind the shader you're about to assemble */
    glBindProgramARB(GL_VERTEX_PROGRAM_ARB, Shader[i]);
    glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			  strlen(ShaderSrc[i]), ShaderSrc[i]);
    finish = microsecs();
    printf("Elapsed = 0.%06d\n", (int)(finish - start));
    glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
    if (err >= 0) {
      getErrorLine(ShaderSrc[i], err, msg, sizeof(msg));
      printf("Error in shader #%d: %s\n", i,
      		glGetString(GL_PROGRAM_ERROR_STRING_ARB));
      printf("Source line:    %s\n", msg);
    }
  }
  
  glBindProgramARB(GL_VERTEX_PROGRAM_NV, Shader[0]);
  				
  glDisable(GL_VERTEX_PROGRAM_ARB);
}

static void initApp ()
{
  TimeBase = time(NULL);
  
  initGraphics();
  initSphere();
  initShader();
  /* Make sure we did everything right */
  CheckGL();
  
  LightPos[0] = 0;
  LightPos[1] = 1;
  LightPos[2] = 4;
  LightPos[3] = 0;
   
  Height     = 1;
  BounceStep = 0.05;
    
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Vertex shader", cmdToggleShader);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  /* Bounce the ball up or down */
  if (Height < -2) {
    Height = -2;
    BounceStep = - BounceStep;
  } else if (Height > 3) {
    Height = 3;
    BounceStep = - BounceStep;
  } else
    Height += BounceStep;

  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("ARB vertex program");

  initApp();
  
  printf("\n");
  printf("V toggles vertex shader\n");
  printf("- and = move light left or right\n");
  printf("M increases shininess\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
