
/*	Hugh Fisher 2003.

	Demonstrate very simple Cg program
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include <GL/glut.h>

#include <Cg/cg.h>
#include <Cg/cgGL.h>

#include "utility.h"
#include "glUtils.h"

#define Near		 1
#define Far		24

#define cmdToggleVertex		1
#define cmdToggleFragment	2
#define cmdSphere		3
#define cmdTeapot		4
#define cmdExit			99

static int	Menu;

static int	UseShader;
static int	Teapot;
static long	TimeBase;

static GLUquadricObj * QState;

static CGcontext	CTX;
static CGprogram	VertShader;
static CGprogram	FragShader;
static CGparameter	CTM;
static CGprofile	VertProf = CG_PROFILE_VP20;
static CGprofile	FragProf = CG_PROFILE_FP30;

static char * VertSrc =
"\
void simpleTL (float4 vPos	: POSITION,	\n\
		 float4 vColor	: COLOR,	\n\
	     out float4 oPos	: POSITION,	\n\
	     out float4 oColor	: COLOR0,	\n\
		uniform float4x4 CTM)		\n\
{						\n\
  oPos   = mul(CTM, vPos);			\n\
  oColor = vColor.zxyw;				\n\
}						\n\
"
;

static char * FragSrc =
"\
void recolor (float4 fColor	: COLOR,	\n\
	  out float4 oColor	: COLOR0)	\n\
{						\n\
  oColor = fColor.gbra;				\n\
}						\n\
"
;


/****		Drawing the world		****/

static void chooseShader (int status)
{
  UseShader = status;
  cgGLDisableProfile(VertProf);  
  cgGLDisableProfile(FragProf);  
  switch (UseShader) {
    case 0:
      printf("Standard OpenGL\n");
      break;
    case 1:
      printf("Vertex shader\n");
      cgGLEnableProfile(VertProf);
      break;
    case 2:
      printf("Fragment shader\n");
      cgGLEnableProfile(FragProf);
      break;
  }
}


static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  gluLookAt(0, 2, 8,
  		0, 0, 0,
		0, 1, 0);
  
  if (UseShader) {
    cgGLSetStateMatrixParameter(CTM,
  	CG_GL_MODELVIEW_PROJECTION_MATRIX,
	CG_GL_MATRIX_IDENTITY);
  }
  
  glColor3f(0, 1, 0);
  if (Teapot)
    glutWireTeapot(1);
  else
    gluSphere(QState, 1, 16, 8);
    
  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdExit:
      exit(0);
      break;
    case cmdToggleVertex:
      if (UseShader == 1)
        chooseShader(0);
      else
        chooseShader(1);
      break;
    case cmdToggleFragment:
      if (UseShader == 2)
        chooseShader(0);
      else
        chooseShader(2);
      break;
    case cmdSphere:
      Teapot = FALSE;
      break;
    case cmdTeapot:
      Teapot = TRUE;
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleVertex);
  else if (key == 'F')
    menuChoice(cmdToggleFragment);
  else if (key == 'S')
    menuChoice(cmdSphere);
  else if (key == 'T')
    menuChoice(cmdTeapot);
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0); 
  glEnable(GL_DEPTH_TEST);
  glShadeModel(GL_SMOOTH);
}

static void initSphere ()
{
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_LINE);
  gluQuadricOrientation(QState, GLU_OUTSIDE);
  gluQuadricNormals(QState, GLU_SMOOTH);
}

static int microsecs ()
{
  struct timeval t;

  gettimeofday(&t, NULL);
  return (t.tv_sec - TimeBase) * 1000000 + t.tv_usec;
}

static void initShaders ()
{
  long start, finish;
  
  CTX = cgCreateContext();
  FailNull(CTX, "context");
  
  cgGLEnableProfile(VertProf);  
  start = microsecs();
  VertShader = cgCreateProgram(CTX, CG_SOURCE, VertSrc,
  		VertProf, "simpleTL", NULL);
  if (! VertShader) {
    printf("Compiler error\n%s\n", cgGetLastListing(CTX));
    Fail("Unable to create vertex shader");
  }
  cgGLLoadProgram(VertShader);
  cgGLBindProgram(VertShader);
  finish = microsecs();
  printf("Elapsed = 0.%06d\n", (int)(finish - start));
  
  CTM = cgGetNamedParameter(VertShader, "CTM");
  FailNull(CTM, "Cannot get CTM parameter");
  
  cgGLEnableProfile(FragProf);  
  start = microsecs();
  FragShader = cgCreateProgram(CTX, CG_SOURCE, FragSrc,
  		FragProf, "recolor", NULL);
  if (! FragShader) {
    printf("Compiler error\n%s\n", cgGetLastListing(CTX));
    Fail("Unable to create fragment shader");
  }
  cgGLLoadProgram(FragShader);
  cgGLBindProgram(FragShader);
  finish = microsecs();
  printf("Elapsed = 0.%06d\n", (int)(finish - start));
}

static void initApp ()
{
  TimeBase = time(NULL);
  
  initGraphics();
  initSphere();
  initShaders();
  /* Make sure we did everything right */
  CheckGL();
  chooseShader(0);
  
  Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Vertex shader", cmdToggleVertex);
  glutAddMenuEntry("Fragment shader", cmdToggleFragment);
  glutAddMenuEntry("Sphere", cmdSphere);
  glutAddMenuEntry("Teapot", cmdTeapot);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);
}

static void animate ()
{
  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("Cg shaders");

  initApp();
  
  printf("\n");
  printf("V toggles vertex shader\n");
  printf("T toggles teapot\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
