
/*	Hugh Fisher 2003.

	Demonstrate very simple ARB vertex/fragment shaders.
	Program displays a sphere or teapot: switching on
	the vertex or fragment shader will adjust the colors
	passed by the program code.
							*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <GL/glext.h>

#include "utility.h"
#include "glUtils.h"

#define Near		-1
#define Far		 1

#define cmdToggleVertex		1
#define cmdToggleFragment	2
#define cmdSphere		3
#define cmdTeapot		4
#define cmdExit			99

static int	Menu;

static int	UseShader;	/* 0 = none 1 = vertex 2 = fragment */
static int	Teapot;		/* true or false */

static GLUquadricObj * QState;

static GLuint VertShader, FragShader;

static char * VertSrc = 
"!!ARBvp1.0						\n\
PARAM	CTM[4]	= { state.matrix.mvp };			\n\
## Standard transform					\n\
DP4  result.position.x, CTM[0], vertex.position;	\n\
DP4  result.position.y, CTM[1], vertex.position;	\n\
DP4  result.position.z, CTM[2], vertex.position;	\n\
DP4  result.position.w, CTM[3], vertex.position;	\n\
# Rotate color	values					\n\
MOV  result.color, vertex.color.zxyw;			\n\
END\n"
;

static char * FragSrc = 
"!!ARBfp1.0						\n\
# Rotate color	values					\n\
MOV  result.color, fragment.color.yzxw;			\n\
END\n"
;


static void chooseShader (int status)
{
  UseShader = status;
  glDisable(GL_VERTEX_PROGRAM_ARB);
  glDisable(GL_FRAGMENT_PROGRAM_ARB);
  switch (UseShader) {
    case 0:
      printf("Standard OpenGL\n");
      break;
    case 1:
      printf("Vertex shader\n");
      glEnable(GL_VERTEX_PROGRAM_ARB);
      break;
    case 2:
      printf("Fragment shader\n");
      glEnable(GL_FRAGMENT_PROGRAM_ARB);
      break;
  }
}

static void display ()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glLoadIdentity();
  
  gluLookAt(0, 2, 8,
  		0, 0, 0,
		0, 1, 0);
  
  glColor3f(0, 1, 0);
  if (Teapot)
    glutWireTeapot(1);
  else
    gluSphere(QState, 1, 16, 8);

  glPopMatrix();  
  CheckGL();
  glutSwapBuffers();
}

static void resize (int width, int height)
{
  GLfloat aspect;

  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, width, height);
  glLoadIdentity();
  aspect = (float)width / (float)height;
  gluPerspective(60.0, aspect, Near, Far);
  glMatrixMode(GL_MODELVIEW);
}


/****		Input handlers		****/


static void menuChoice (int item)
{
  switch (item) {
    case cmdToggleVertex:
      if (UseShader == 1)
        chooseShader(0);
      else
        chooseShader(1);
      break;
    case cmdToggleFragment:
      if (UseShader == 2)
        chooseShader(0);
      else
        chooseShader(2);
      break;
    case cmdSphere:
      Teapot = FALSE;
      break;
    case cmdTeapot:
      Teapot = TRUE;
      break;
    case cmdExit:
      exit(0);
      break;
    default:
      break;
  }
}

static void asciiKey (unsigned char key, int x, int y)
{
  key = toupper(key);
  
  if (key == 27) /* ESC */
    exit(0);
  else if (key == 'V')
    menuChoice(cmdToggleVertex);
  else if (key == 'F')
    menuChoice(cmdToggleFragment);
  else if (key == 'S')
    menuChoice(cmdSphere);
  else if (key == 'T')
    menuChoice(cmdTeapot);
}


/****		Main control		****/


static void initGraphics ()
{
  glClearColor(0, 0, 0, 0); 
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);
  glShadeModel(GL_SMOOTH);
  
  QState = gluNewQuadric();
  FailNull(QState, "Cannot allocate quadric object");
  
  gluQuadricDrawStyle(QState, GLU_LINE);
}

static void initShaders ()
{
  int err;
  
  if (! glutExtensionSupported("GL_ARB_vertex_program"))
    Fail("GL_ARB_vertex_program not available on this machine");
  if (! glutExtensionSupported("GL_ARB_fragment_program"))
    Fail("GL_ARB_fragment_program not available on this machine");
  
  /* Vertex shader */
  glEnable(GL_VERTEX_PROGRAM_ARB);
  glGenProgramsARB(1, &VertShader);
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, VertShader);
  glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			  strlen(VertSrc), VertSrc);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0)
    printf("Error in vertex shader %s\n",
    		glGetString(GL_PROGRAM_ERROR_STRING_ARB));
  
  /* Fragment shader */
  glEnable(GL_FRAGMENT_PROGRAM_ARB);
  glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, &FragShader);
  glProgramStringARB(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
  			strlen(FragSrc), FragSrc);
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &err);
  if (err >= 0)
    printf("Error in fragment shader %s\n",
    		glGetString(GL_PROGRAM_ERROR_STRING_ARB));
}

static void initApp ()
{
  initGraphics();
  initShaders();
  /* Make sure we did everything right */
  CheckGL();
  chooseShader(0);
  
   Menu = glutCreateMenu(menuChoice);
  glutSetMenu(Menu);
  glutAddMenuEntry("Vertex shader", cmdToggleVertex);
  glutAddMenuEntry("Fragment shader", cmdToggleFragment);
  glutAddMenuEntry("Sphere", cmdSphere);
  glutAddMenuEntry("Teapot", cmdTeapot);
  glutAddMenuEntry("----", 0);
  glutAddMenuEntry("Exit", cmdExit);
  glutAttachMenu(GLUT_RIGHT_BUTTON);  
}

static void animate ()
{
  glutPostRedisplay();
}

int main (int argc, char * argv[])
{
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);

  glutInitWindowSize(800, 600);
  glutInitWindowPosition(100, 75);
  glutCreateWindow("ARB shaders");

  initApp();
  
  printf("\n");
  printf("V toggles use of vertex shader\n");
  printf("F toggles use of fragment shader\n");
  printf("T draws teapot\n");
  printf("S draws sphere\n");
  printf("\n");
  
  glutDisplayFunc(display);
  glutReshapeFunc(resize);
  glutIdleFunc(animate);
  glutKeyboardFunc(asciiKey);
  
  glutMainLoop();
  return 0;	/* Keeps compiler happy */
}
